﻿//=============================================================================
// Blur.fx Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// Dokonuje separowalnego rozmycia o promieniu 5.  
//=============================================================================

cbuffer cbSettings
{
	float gWeights[11] = 
	{
		0.05f, 0.05f, 0.1f, 0.1f, 0.1f, 0.2f, 0.1f, 0.1f, 0.1f, 0.05f, 0.05f,
	};
};

cbuffer cbFixed
{
	static const int gBlurRadius = 5;
};

Texture2D gInput;
RWTexture2D<float4> gOutput;

#define N 256
#define CacheSize (N + 2*gBlurRadius)
groupshared float4 gCache[CacheSize];

[numthreads(N, 1, 1)]
void HorzBlurCS(int3 groupThreadID : SV_GroupThreadID,
				int3 dispatchThreadID : SV_DispatchThreadID)
{
	//
    // Wypełnij lokalną pamięć wątku, aby ograniczyć ilość przesyłanych danych. Aby rozmyć 
    // N pikseli, potrzebujemy załadować N + 2*BlurRadius pikseli
    // ze względu na promieĹ„ rozmycia.
	//
	
    // Ta grupa uruchamia N wątków. Aby uzyskać dodatkowe 2*BlurRadius
    // pikseli, 2*BlurRadius wątków próbkuje dodatkowy piksel.
	if(groupThreadID.x < gBlurRadius)
	{
		// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
		int x = max(dispatchThreadID.x - gBlurRadius, 0);
		gCache[groupThreadID.x] = gInput[int2(x, dispatchThreadID.y)];
	}
	if(groupThreadID.x >= N-gBlurRadius)
	{
		// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
		int x = min(dispatchThreadID.x + gBlurRadius, gInput.Length.x-1);
		gCache[groupThreadID.x+2*gBlurRadius] = gInput[int2(x, dispatchThreadID.y)];
	}

	// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
	gCache[groupThreadID.x+gBlurRadius] = gInput[min(dispatchThreadID.xy, gInput.Length.xy-1)];

	// Czekaj na zakoĹ„czenie wszystkich wątków.
	GroupMemoryBarrierWithGroupSync();
	
	//
	// Rozmyj każdy piksel.
	//

	float4 blurColor = float4(0, 0, 0, 0);
	
	[unroll]
	for(int i = -gBlurRadius; i <= gBlurRadius; ++i)
	{
		int k = groupThreadID.x + gBlurRadius + i;
		
		blurColor += gWeights[i+gBlurRadius]*gCache[k];
	}
	
	gOutput[dispatchThreadID.xy] = blurColor;
}

[numthreads(1, N, 1)]
void VertBlurCS(int3 groupThreadID : SV_GroupThreadID,
				int3 dispatchThreadID : SV_DispatchThreadID)
{
	//
    // Wypełnij lokalną pamięć wątku, aby ograniczyć ilość przesyłanych danych. Aby rozmyć 
    // N pikseli, potrzebujemy załadować N + 2*BlurRadius pikseli
    // ze względu na promieĹ„ rozmycia.
	//
	
   // Ta grupa uruchamia N wątków. Aby uzyskać dodatkowe 2*BlurRadius 
   // pikseli, 2*BlurRadius wątków próbkuje dodatkowy piksel.
	if(groupThreadID.y < gBlurRadius)
	{
		// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
		int y = max(dispatchThreadID.y - gBlurRadius, 0);
		gCache[groupThreadID.y] = gInput[int2(dispatchThreadID.x, y)];
	}
	if(groupThreadID.y >= N-gBlurRadius)
	{
		// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
		int y = min(dispatchThreadID.y + gBlurRadius, gInput.Length.y-1);
		gCache[groupThreadID.y+2*gBlurRadius] = gInput[int2(dispatchThreadID.x, y)];
	}
	
	// Obetnij próbki spoza granic obrazu pojawiające się na jego brzegach.
	gCache[groupThreadID.y+gBlurRadius] = gInput[min(dispatchThreadID.xy, gInput.Length.xy-1)];


	// Czekaj na zakoĹ„czenie wszystkich wątków.
	GroupMemoryBarrierWithGroupSync();
	
	//
	// Rozmyj każdy piksel.
	//

	float4 blurColor = float4(0, 0, 0, 0);
	
	[unroll]
	for(int i = -gBlurRadius; i <= gBlurRadius; ++i)
	{
		int k = groupThreadID.y + gBlurRadius + i;
		
		blurColor += gWeights[i+gBlurRadius]*gCache[k];
	}
	
	gOutput[dispatchThreadID.xy] = blurColor;
}

technique11 HorzBlur
{
    pass P0
    {
		SetVertexShader( NULL );
        SetPixelShader( NULL );
		SetComputeShader( CompileShader( cs_5_0, HorzBlurCS() ) );
    }
}

technique11 VertBlur
{
    pass P0
    {
		SetVertexShader( NULL );
        SetPixelShader( NULL );
		SetComputeShader( CompileShader( cs_5_0, VertBlurCS() ) );
    }
}
